from fastapi import FastAPI, Form, Request, Depends, HTTPException
from fastapi.templating import Jinja2Templates
from fastapi.responses import HTMLResponse, RedirectResponse
from sqlalchemy import create_engine, Column, Integer, String, ForeignKey, Date, Float, desc, case
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy.orm import sessionmaker, joinedload
from sqlalchemy.sql import func
from datetime import date
from passlib.context import CryptContext
from starlette.middleware.sessions import SessionMiddleware
import secrets
from datetime import date


# Konfiguracja bezpieczeństwa
pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")
app = FastAPI()
app.add_middleware(SessionMiddleware, secret_key=secrets.token_urlsafe(32))

# Konfiguracja bazy danych
DATABASE_URL = "sqlite:///./sprawpol.db"
engine = create_engine(DATABASE_URL, connect_args={"check_same_thread": False}, pool_pre_ping=True)
SessionLocal = sessionmaker(autocommit=False, autoflush=False, bind=engine)
Base = declarative_base()

# Modele
class Uzytkownicy(Base):
    __tablename__ = "Uzytkownicy"
    id_uzytkownika = Column(Integer, primary_key=True)
    login = Column(String(90), unique=True)
    haslo = Column(String(90))

class Prowadzacy(Base):
    __tablename__ = "Prowadzacy"
    id_prowadzacego = Column(Integer, primary_key=True)
    imie = Column(String(25))
    nazwisko = Column(String(40))

class Przedmioty(Base):
    __tablename__ = "Przedmioty"
    id_przedmiotu = Column(Integer, primary_key=True)
    nazwa = Column(String(90))

class Przypisania(Base):
    __tablename__ = "Przypisania"
    id_przypisania = Column(Integer, primary_key=True)
    id_prowadzacego = Column(Integer, ForeignKey("Prowadzacy.id_prowadzacego"))
    id_przedmiotu = Column(Integer, ForeignKey("Przedmioty.id_przedmiotu"))

class Sprawozdania(Base):
    __tablename__ = "Sprawozdania"
    id_sprawozdania = Column(Integer, primary_key=True)
    id_przypisania = Column(Integer, ForeignKey("Przypisania.id_przypisania"))
    id_uzytkownika = Column(Integer, ForeignKey("Uzytkownicy.id_uzytkownika"))
    tytul = Column(String(100))
    ocena = Column(Float)
    poprawiane = Column(Integer)
    data_oddania = Column(Date)
    termin = Column(Date)

Base.metadata.create_all(bind=engine)
templates = Jinja2Templates(directory="templates")

# Dependency
def get_db():
    db = SessionLocal()
    try:
        yield db
    finally:
        db.close()

# Funkcje pomocnicze
def verify_password(plain, hashed):
    return pwd_context.verify(plain, hashed)

def get_hash(password):
    return pwd_context.hash(password)

# Endpointy
@app.get("/", response_class=HTMLResponse)
async def login_page(request: Request):
    return templates.TemplateResponse("login.html", {"request": request})

@app.post("/login/")
async def login(
    request: Request,
    login: str = Form(...),
    haslo: str = Form(...),
    db: SessionLocal = Depends(get_db)
):
    user = db.query(Uzytkownicy).filter(Uzytkownicy.login == login).first()
    if not user or not verify_password(haslo, user.haslo):
        return templates.TemplateResponse("login.html", 
            {"request": request, "error": "Nieprawidłowe dane logowania"})
    
    request.session["user_id"] = user.id_uzytkownika
    request.session["login"] = user.login
    return RedirectResponse(url="/sprawozdania", status_code=302)

@app.post("/register/")
async def register(
    request: Request,
    new_login: str = Form(...),
    new_password: str = Form(...),
    db: SessionLocal = Depends(get_db)
):
    existing = db.query(Uzytkownicy).filter(Uzytkownicy.login == new_login).first()
    if existing:
        return templates.TemplateResponse("login.html",
            {"request": request, "error": "Login już istnieje"})
    
    new_user = Uzytkownicy(
        login=new_login,
        haslo=get_hash(new_password)
    )
    db.add(new_user)
    db.commit()
    return RedirectResponse(url="/?registered=1", status_code=302)

@app.get("/sprawozdania", response_class=HTMLResponse)
async def show_reports(request: Request, db: SessionLocal = Depends(get_db)):
    user_id = request.session.get("user_id")
    if not user_id:
        return RedirectResponse(url="/")
    
    sort = request.query_params.get("sort", "data_oddania")
    allowed_sorts = {
        "data_oddania": Sprawozdania.data_oddania,
        "ocena": Sprawozdania.ocena,
        "tytul": Sprawozdania.tytul
    }
    
    reports = (
        db.query(Sprawozdania, Przedmioty.nazwa, Prowadzacy.imie, Prowadzacy.nazwisko)
        .join(Przypisania, Przypisania.id_przypisania == Sprawozdania.id_przypisania)
        .join(Przedmioty, Przedmioty.id_przedmiotu == Przypisania.id_przedmiotu)
        .join(Prowadzacy, Prowadzacy.id_prowadzacego == Przypisania.id_prowadzacego)
        .filter(Sprawozdania.id_uzytkownika == user_id)
        .order_by(desc(allowed_sorts.get(sort, Sprawozdania.data_oddania)))
        .all()
    )
    
    przypisania = (
        db.query(Przypisania, Przedmioty.nazwa, Prowadzacy.imie, Prowadzacy.nazwisko)
        .join(Przedmioty, Przedmioty.id_przedmiotu == Przypisania.id_przedmiotu)
        .join(Prowadzacy, Prowadzacy.id_prowadzacego == Przypisania.id_prowadzacego)
        .all()
    )
    
    return templates.TemplateResponse("sprawozdania.html", {
        "request": request,
        "reports": reports,
        "przypisania": przypisania,
        "sort": sort
    })

@app.post("/add-report/")
async def add_report(
    request: Request,
    tytul: str = Form(...),
    id_przypisania: int = Form(...),
    termin: str = Form(...),
    db: SessionLocal = Depends(get_db)
):
    
    user_id = request.session.get("user_id")
    if not user_id:
        return RedirectResponse(url="/")

    new_report = Sprawozdania(
        id_przypisania=id_przypisania,
        id_uzytkownika=user_id,
        tytul=tytul,
        data_oddania=date.today(),
	termin=date.fromisoformat(termin),
        poprawiane=0
    )
    
    db.add(new_report)
    db.commit()
    return RedirectResponse(url="/sprawozdania", status_code=302)

@app.post("/add-przypisanie/")
async def add_przypisanie(
    request: Request,
    id_przedmiotu: int = Form(...),
    id_prowadzacego: int = Form(...),
    db: SessionLocal = Depends(get_db)
):

    new_przypisanie = Przypisania(
	id_przedmiotu = id_przedmiotu,
	id_prowadzacego = id_prowadzacego
    )

    db.add(new_przypisanie)
    db.commit()
    return RedirectResponse(url="/nowy-przedmiot", status_code=302)


@app.post("/add-przedmiot/")
async def add_przedmiot(
    request: Request,
    nazwa: str = Form(...),
    db: SessionLocal = Depends(get_db)
):

    new_przedmiot = Przedmioty(
        nazwa = nazwa
    )

    db.add(new_przedmiot)
    db.commit()
    return RedirectResponse(url="/nowy-przedmiot", status_code=302)


@app.post("/add-prowadzacy/")
async def add_prowadzacy(
    request: Request,
    imie: str = Form(...),
    nazwisko: str = Form(...),
    db: SessionLocal = Depends(get_db)
):

    new_prowadzacy = Prowadzacy(
        imie = imie,
	nazwisko = nazwisko
    )

    db.add(new_prowadzacy)
    db.commit()
    return RedirectResponse(url="/nowy-przedmiot", status_code=302)


@app.get("/nowy-przedmiot")
async def add_subject(
    request: Request,
    db: SessionLocal = Depends(get_db)

):

    user_id = request.session.get("user_id")
    if not user_id:
        return RedirectResponse(url="/")


    prowadzacy = (
        db.query(Prowadzacy.imie, Prowadzacy.nazwisko, Prowadzacy.id_prowadzacego)
        .all()
    )

    przedmioty = (
        db.query(Przedmioty.nazwa, Przedmioty.id_przedmiotu)
       	.all()
    )


    return templates.TemplateResponse("nowy-przedmiot.html", 
	{"request": request,
	 "prowadzacy": prowadzacy,
	 "przedmioty": przedmioty,

	})


@app.post("/edit-report/{report_id}")
async def edit_report(
    request: Request,
    report_id: int,
    ocena: str = Form(...),
    poprawiane: int = Form(...),
    db: SessionLocal = Depends(get_db)
):
    user_id = request.session.get("user_id")
    report = db.query(Sprawozdania).filter(
        Sprawozdania.id_sprawozdania == report_id,
        Sprawozdania.id_uzytkownika == user_id
    ).first()
    
    if report:
        report.ocena = ocena
        report.poprawiane = poprawiane if poprawiane > 0 else 0
        db.commit()
    
    return RedirectResponse(url="/sprawozdania", status_code=302)

@app.get("/delete-report/{report_id}")
async def delete_report(
    request: Request,
    report_id: int,
    db: SessionLocal = Depends(get_db)
):
    user_id = request.session.get("user_id")
    report = db.query(Sprawozdania).filter(
        Sprawozdania.id_sprawozdania == report_id,
        Sprawozdania.id_uzytkownika == user_id
    ).first()
    
    if report:
        db.delete(report)
        db.commit()
    
    return RedirectResponse(url="/sprawozdania", status_code=302)

@app.get("/podsumowanie")
async def summary(
    request: Request,
    db: SessionLocal = Depends(get_db)
):

    user_id = request.session.get("user_id")
    if not user_id:
        return RedirectResponse(url="/")


    results = (
        db.query(
            Przedmioty.nazwa.label("przedmiot"),
            func.avg(Sprawozdania.ocena).label("srednia_ocena"),
            func.count(
                case(
                    [(Sprawozdania.ocena < 3.0, 1)],
		    else_=None
                )
            ).label("below3")
        )
        .join(Przypisania, Sprawozdania.id_przypisania == Przypisania.id_przypisania)
        .join(Przedmioty, Przypisania.id_przedmiotu == Przedmioty.id_przedmiotu)
        .filter(Sprawozdania.id_uzytkownika == user_id)
        .group_by(Przedmioty.nazwa)
        .all()
    )

    return templates.TemplateResponse("podsumowanie.html", {
        "results": results,
	"request": request
    })


@app.get("/logout")
async def logout(request: Request):
    request.session.clear()
    return templates.TemplateResponse("login.html",
            {"request": request, "error": "Wylogowano poprawnie"})
